<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Kiwi_Social_Share_Settings
 */
class Kiwi_Social_Share_Settings {

	/**
	 * The single instance of Kiwi_Social_Share_Settings.
	 *
	 * @var    object
	 * @access   private
	 * @since    1.0.0
	 */
	private static $_instance = NULL;

	/**
	 * The main plugin object.
	 *
	 * @var    object
	 * @access   public
	 * @since    1.0.0
	 */
	public $parent = NULL;

	/**
	 * Prefix for plugin settings.
	 *
	 * @var     string
	 * @access  public
	 * @since   1.0.0
	 */
	public $base = '';

	/**
	 * Available settings for plugin.
	 *
	 * @var     array
	 * @access  public
	 * @since   1.0.0
	 */
	public $settings = array();

	/**
	 * Kiwi_Social_Share_Settings constructor.
	 *
	 * @param $parent
	 */
	public function __construct( $parent ) {
		$this->parent = $parent;

		$this->base = 'kiwi_';

		// Initialise settings
		add_action( 'init', array( $this, 'init_settings' ), 11 );

		// Register plugin settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Add settings page to menu
		add_action( 'admin_menu', array( $this, 'add_menu_item' ) );

		// Add settings link to plugins page
		add_filter( 'plugin_action_links_' . plugin_basename( $this->parent->file ), array(
			$this,
			'add_settings_link'
		) );
	}

	/**
	 * Initialise settings
	 *
	 * @return void
	 */
	public function init_settings() {
		$this->settings = $this->settings_fields();
	}

	/**
	 * Add settings page to admin menu
	 *
	 * @return void
	 */
	public function add_menu_item() {
		/* start-lite-version */
		$page = add_menu_page( __( 'Kiwi', 'kiwi-social-share' ), __( 'Kiwi', 'kiwi-social-share' ), 'manage_options', $this->parent->_token . '_settings', array(
			$this,
			'settings_page'
		), 'dashicons-share-alt' );
		/* end-lite-version */


		$advanced_shortcodes = Kiwi_Social_Share_Helper::get_setting_value( 'advanced_shortcode_manager', false, 'kiwi_advanced_settings' );
		if ( $advanced_shortcodes ) {
			add_submenu_page( $this->parent->_token . '_settings', __( 'Shortcodes', 'kiwi-social-share' ), __( 'Shortcodes', 'kiwi-social-share' ), 'manage_options', 'edit.php?post_type=kiwi-shortcodes' );
		}
	}

	/**
	 * Add settings link to plugin list table
	 *
	 * @param  array $links Existing links
	 *
	 * @return array        Modified links
	 */
	public function add_settings_link( $links ) {
		$settings_link = '<a href="options-general.php?page=' . $this->parent->_token . '_settings">' . esc_html__( 'Settings', 'kiwi-social-share' ) . '</a>';
		array_push( $links, $settings_link );

		return $links;
	}

	/**
	 * Build settings fields
	 *
	 * @return array Fields to be displayed on settings page
	 */
	private function settings_fields() {

		$settings['standard'] = array(
			'title'  => '',
			'fields' => array(
				array(
					'id'       => 'general_settings',
					'callback' => array( 'Kiwi_Social_Share_Sanitizers', 'automatic' ),
				),

				array(
					'id'       => 'network_colors',
					'callback' => array( 'Kiwi_Social_Share_Sanitizers', 'multiple_hex_fields' ),
				),

				array(
					'id'       => 'social_identities',
					'callback' => array( 'Kiwi_Social_Share_Sanitizers', 'automatic' ),
				),

				array(
					'id'       => 'advanced_settings',
					'callback' => array( 'Kiwi_Social_Share_Sanitizers', 'automatic' ),
				),
			)
		);

		$settings = apply_filters( $this->parent->_token . '_settings_fields', $settings );

		return $settings;
	}

	/**
	 * Register plugin settings
	 *
	 * @return void
	 */
	public function register_settings() {
		if ( is_array( $this->settings ) ) {

			// Check posted/selected tab
			$current_section = '';
			if ( isset( $_POST['tab'] ) && $_POST['tab'] ) {
				$current_section = $_POST['tab'];
			} else {
				if ( isset( $_GET['tab'] ) && $_GET['tab'] ) {
					$current_section = $_GET['tab'];
				}
			}

			foreach ( $this->settings as $section => $data ) {

				if ( $current_section && $current_section != $section ) {
					continue;
				}

				// Add section to page
				add_settings_section( $section, $data['title'], array(), $this->parent->_token . '_settings' );

				foreach ( $data['fields'] as $field ) {

					// Validation callback for field
					$validation = '';
					if ( isset( $field['callback'] ) ) {
						$validation = $field['callback'];
					}

					// Register field
					$option_name = $this->base . $field['id'];
					register_setting( $this->parent->_token . '_settings', $option_name, $validation );
				}

				if ( ! $current_section ) {
					break;
				}
			}
		}
	}

	/**
	 * Load settings page content
	 *
	 * @return void
	 */
	public function settings_page() {
		require_once KIWI_SOCIAL_SHARE_BASE . '/includes/backend/kiwi-social-share-backend.php';
	}

	/**
	 * Main Kiwi_Social_Share_Settings Instance
	 *
	 * Ensures only one instance of Kiwi_Social_Share_Settings is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 * @see   Kiwi_Social_Share()
	 * @return Main Kiwi_Social_Share_Settings instance
	 */
	public static function instance( $parent ) {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self( $parent );
		}

		return self::$_instance;
	} // End instance()

	/**
	 * Cloning is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?' ), $this->parent->_version );
	} // End __clone()

	/**
	 * Unserializing instances of this class is forbidden.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?' ), $this->parent->_version );
	} // End __wakeup()

}